suppressWarnings(suppressMessages(library(spatstat)))
suppressWarnings(library(FNN))

create.point.pattern <- function(df, xrange, yrange, types) {
    if (!all(c("x", "y") %in% colnames(df))) {
        stop("Dataframe must include columns 'x' and 'y'.")
    }
    if ((xrange[1] >= xrange[2]) | (yrange[1] >= yrange[2])) {
        stop("xrange or yrange not corret.")
    }
    if (!("group" %in% colnames(df))) {
        stop("Dataframe must include column 'group' for subset points.")
    }
    if (sum(is.na(df$group)) > 0) {
        warning("NA type found, removed for following analysis.")
        df <- df[!is.na(df$group),]
    }
    df <- df[(df$x >= xrange[1]) & (df$x <= xrange[2]) & (df$y >= yrange[1]) & (df$y <= yrange[2]),]
    if (nrow(df) < 10) {
        stop("Too few point in the pattern.")
    }
    if (!is.character(types)) {
        stop("Only accept character types.")
    }
    if (length(types) > 6) {
        stop("Accept maximum 6 types.")
    }
    for (i in types) {
        if (sum(df$group == i) == 0) {
            stop(paste("Type", i, "not found.", sep = " "))
        }
    }
    df$group <- as.character(df$group)
    df <- df[df$group %in% types,]
    m <- factor(df$group)
    print(paste("Types in the region:", paste(levels(m), collapse = " "), sep = " "))
    yrange = c(-yrange[2], -yrange[1])
    return(ppp(x = df$x, y = -df$y, window = owin(xrange = xrange, yrange = yrange), marks = m))
}

plot.markcorr.pair <- function(spat.data, path) {
    types <- levels(spat.data$marks)
    n <- length(types)
    par.old <- par()
    png(path, width = 900 * n, height = 900 * n, res = 180, units = "px", type = "cairo")
    par(mfrow = c(n, n), pty = "s", mar = c(3,3,3,3))
    for (i in 1:n) {
        for (j in 1:n) {
            if (i == j) {
                plot.density(spat.data, types[i], main = paste("Type", types[i]))
            } else if (i > j) {
                par(mar = c(4,4,4,4))
                plot.markcorr(spat.data, c(types[i], types[j]))
                par(mar = c(3,3,3,3))
            } else {
                plot.new()
            }
        }
    }
    dev.off()
    par(par.old)
}

find.nn <- function(spat.data, out.path) {
    # nearest neghbors
    df <- data.frame(x = spat.data$x, y = spat.data$y, group = spat.data$marks)
    nn <- get.knn(df[,c("x", "y")], k = 1)
    nn <- data.frame(group = df$group, nn.index = nn$nn.index, nn.dist = nn$nn.dist, nn.group = df$group[nn$nn.index])
    all.labels <- levels(nn$group)
    nn.count <- data.frame(matrix(0, length(all.labels), length(all.labels)), row.names = all.labels)
    colnames(nn.count) <- all.labels
    for (i in all.labels) {
        tmp <- nn[nn$group == i,]
        count.tmp <- aggregate(list(prop = tmp$nn.index), by = list(group = tmp$nn.group), FUN = length)
        nn.count[i, as.character(count.tmp$group)] <- count.tmp$prop
    }
    write.csv(nn.count, out.path)
    return(nn.count)
}

compare.markcorr <- function(spat.data.1, spat.data.2, path) {
    types <- levels(spat.data.1$marks)
    par.old <- par()
    png(path, width = 2700, height = 1800, res = 180, units = "px", type = "cairo")
    par(mfrow = c(2, 3), pty = "s", mar = c(3,3,3,3))
    plot.density(spat.data.1, types[1], main = paste("Region A type", types[1]))
    plot.density(spat.data.1, types[2], main = paste("Region A type", types[2]))
    par(mar = c(4,4,4,4))
    plot.markcorr(spat.data.1, types)
    par(mar = c(3,3,3,3))
    plot.density(spat.data.2, types[1], main = paste("Region B type", types[1]))
    plot.density(spat.data.2, types[2], main = paste("Region B type", types[2]))
    par(mar = c(4,4,4,4))
    plot.markcorr(spat.data.2, types)
    par(par.old)
    dev.off()
}

test.nn <- function(table.1, table.2, path) {
    y <- c(unlist(table.1), unlist(table.2))
    p <- poisson.glm(y)
    f <- file(path)
    writeLines(as.character(p), f)
    close(f)
}

### ----------------  utility function  ------------------ ###
plot.density <- function(spat.data, type, main = "") {
    spat.data <- unmark(subset.ppp(spat.data, marks == type, drop = TRUE))
    dens <- density(spat.data, kernel = "gaussian", edge = TRUE, diggle = TRUE)
    plot(spat.data, main = main)
    plot(dens, main = "", add = TRUE)
    plot(spat.data, main = "", add = TRUE)
}

plot.markcorr <- function(spat.data, types) {
    if (length(types) != 2) {
        stop("Only accept two types.")
    }
    spat.data <- subset.ppp(spat.data, marks %in% types, drop = TRUE)
    mark.corr.envelope <- envelope(spat.data, markcorr, correction = "none", nsim = 99)
    plot(mark.corr.envelope, main = "", legend = FALSE)
}

pv.expr <- function(x, digits = 1) {
    if (!x) return(0)
    exponent <- floor(log10(x))
    base <- round(x / 10^exponent, digits)
    ifelse(x > 0.000001, as.character(base*(10^exponent)),
           paste(base, "E", exponent, sep=""))
}

poisson.glm <- function(y) {
    if (length(y) != 8) {
        stop("Only deal with 8 numbers.")
    }
    xi <- rep(c(0, 1), 4)
    xj <- rep(c(0, 0, 1, 1), 2)
    xk <- c(rep(0, 4), rep(1, 4))
    m.full <- glm(y ~ (xi + xj + xi*xj) * xk, family = poisson)
    m.null <- glm(y ~ (xi + xj + xi*xj) + xk, family = poisson)
    p <- pchisq(deviance(m.null) - deviance(m.full), df = 3, lower.tail = FALSE)
    p <- pv.expr(p)
    return(p)
}